<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Admin
 * @author  StudioPress
 * @license GPL-2.0+
 * @link    http://my.studiopress.com/themes/genesis/
 */

add_action( 'admin_menu', 'genesis_add_inpost_seo_box' );
/**
 * Register a new meta box to the post or page edit screen, so that the user can set SEO options on a per-post or
 * per-page basis.
 *
 * If the post type does not support genesis-seo, then the SEO meta box will not be added.
 *
 * @since 0.1.3
 *
 * @see genesis_inpost_seo_box() Generates the content in the meta box.
 */
function genesis_add_inpost_seo_box() {

	foreach ( (array) get_post_types( array( 'public' => true ) ) as $type ) {
		if ( post_type_supports( $type, 'genesis-seo' ) ) {
			add_meta_box( 'genesis_inpost_seo_box', __( 'Theme SEO Settings', 'genesis' ), 'genesis_inpost_seo_box', $type, 'normal', 'high' );
		}

	}

	add_action( 'load-post.php', 'genesis_seo_contextual_help' );
	add_action( 'load-post-new.php', 'genesis_seo_contextual_help' );

}

/**
 * Callback for in-post SEO meta box.
 *
 * @since 0.1.3
 */
function genesis_inpost_seo_box() {

	wp_nonce_field( 'genesis_inpost_seo_save', 'genesis_inpost_seo_nonce' );
	?>

	<table class="form-table">
	<tbody>

		<tr valign="top">
			<th scope="row"><label for="genesis_title"><?php _e( 'Document Title', 'genesis' ); ?><span class="screen-reader-text"> &lt;title&gt; Tag. </span></label></th>
			<td>
				<p><input class="large-text" type="text" name="genesis_seo[_genesis_title]" id="genesis_title" value="<?php echo esc_attr( genesis_get_custom_field( '_genesis_title' ) ); ?>" /></p>
				<p><span class="hide-if-no-js description"><?php printf( __( 'Characters Used: %s', 'genesis' ), '<span id="genesis_title_chars">'. mb_strlen( genesis_get_custom_field( '_genesis_title' ) ) .'</span>' ); ?></span></p>
			</td>
		</tr>

		<tr valign="top">
			<th scope="row"><label for="genesis_description"><?php _e( 'Meta Description', 'genesis' ); ?><span class="screen-reader-text"> &lt;meta name=&quot;description&quot; /&gt;. </span></label></th>
			<td>
				<p><textarea class="widefat" name="genesis_seo[_genesis_description]" id="genesis_description" rows="4" cols="4"><?php echo esc_textarea( genesis_get_custom_field( '_genesis_description' ) ); ?></textarea></p>
				<p><span class="hide-if-no-js description"><?php printf( __( 'Characters Used: %s', 'genesis' ), '<span id="genesis_description_chars">'. mb_strlen( genesis_get_custom_field( '_genesis_description' ) ) .'</span>' ); ?></span></p>
			</td>
		</tr>

		<tr valign="top">
			<th scope="row"><label for="genesis_keywords"><strong><?php _e( 'Meta Keywords', 'genesis' ); ?></strong><span class="screen-reader-text"> &lt;meta name=&quot;keywords&quot; /&gt;. </span></label></th>
			<td><p><input class="large-text" type="text" name="genesis_seo[_genesis_keywords]" id="genesis_keywords" value="<?php echo esc_attr( genesis_get_custom_field( '_genesis_keywords' ) ); ?>" /></p></td>
		</tr>

		<tr valign="top">
			<th scope="row"><label for="genesis_canonical"><strong><?php _e( 'Canonical URL', 'genesis' ); ?></strong> <span class="screen-reader-text"> &lt;link rel=&quot;canonical&quot; /&gt;. </span></label></th>
			<td><p><input class="large-text" type="text" name="genesis_seo[_genesis_canonical_uri]" id="genesis_canonical" value="<?php echo esc_url( genesis_get_custom_field( '_genesis_canonical_uri' ) ); ?>" /></p></td>
		</tr>

		<tr valign="top">
			<th scope="row"><label for="genesis_redirect"><strong><?php _e( 'Custom Redirect URL', 'genesis' ); ?></strong><span class="screen-reader-text"> 301 Redirect. </span></label></th>
			<td><p><input class="large-text" type="text" name="genesis_seo[redirect]" id="genesis_redirect" value="<?php echo esc_url( genesis_get_custom_field( 'redirect' ) ); ?>" /></p></td>
		</tr>

		<tr valign="top">
			<th scope="row"><?php _e( 'Robots Meta Settings', 'genesis' ); ?></th>
			<td>
				<p>
					<label for="genesis_noindex"><input type="checkbox" name="genesis_seo[_genesis_noindex]" id="genesis_noindex" value="1" <?php checked( genesis_get_custom_field( '_genesis_noindex' ) ); ?> />
					<?php printf( __( 'Apply %s to this post/page', 'genesis' ), genesis_code( 'noindex' ) ); ?></label><br />
				</p>
				<p>
					<label for="genesis_nofollow"><input type="checkbox" name="genesis_seo[_genesis_nofollow]" id="genesis_nofollow" value="1" <?php checked( genesis_get_custom_field( '_genesis_nofollow' ) ); ?> />
					<?php printf( __( 'Apply %s to this post/page', 'genesis' ), genesis_code( 'nofollow' ) ); ?></label><br />
				</p>
				<p>
					<label for="genesis_noarchive"><input type="checkbox" name="genesis_seo[_genesis_noarchive]" id="genesis_noarchive" value="1" <?php checked( genesis_get_custom_field( '_genesis_noarchive' ) ); ?> />
					<?php printf( __( 'Apply %s to this post/page', 'genesis' ), genesis_code( 'noarchive' ) ); ?>
				</p>
			</td>
		</tr>

	</tbody>
	</table>

	<?php

}

/**
 * Callback for in-post SEO meta box contextual help.
 *
 * @since 2.4.0
 */
function genesis_seo_contextual_help() {

	global $typenow;

	if ( ! post_type_supports( $typenow, 'genesis-seo' ) ) {
		return;
	}

	$content = '<p>' . sprintf( __( '<strong>Document Title</strong> &mdash; Output in the document %s Tag.', 'genesis' ), esc_html( '<title>' ) ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Meta Description</strong> &mdash; Output in the document %s.', 'genesis' ), esc_html( '<meta name="description" />' ) ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Meta Keywords</strong> &mdash; Output in the document %s.', 'genesis' ), esc_html( '<meta name="keywords" />' ) ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Canonical URL</strong> &mdash; Output in the document %s. %s.', 'genesis' ), esc_html( '<link rel="canonical" />' ), '<a href="http://www.mattcutts.com/blog/canonical-link-tag/" target="_blank">' . __( 'Read more about Custom Canonical URL', 'genesis' ) . '</a>' ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Custom Redirect URL</strong> &mdash; Redirect this post/page to this URL. %s.', 'genesis' ), '<a href="http://www.google.com/support/webmasters/bin/answer.py?hl=en&amp;answer=93633" target="_blank">' . __( 'Read more about 301 Redirects', 'genesis' ) . '</a>' ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Apply noindex to this post/page</strong> &mdash; Output in the document %s. %s.', 'genesis' ), esc_html( '<meta name="robots" />' ), '<a href="http://yoast.com/articles/robots-meta-tags/" target="_blank">' . __( 'Read more about noindex', 'genesis' ) . '</a>' ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Apply nofollow to this post/page</strong> &mdash; Output in the document %s. %s.', 'genesis' ), esc_html( '<meta name="robots" />' ), '<a href="http://yoast.com/articles/robots-meta-tags/" target="_blank">' . __( 'Read more about nofollow', 'genesis' ) . '</a>' ) . '</p>';
	$content .= '<p>' . sprintf( __( '<strong>Apply noarchive to this post/page</strong> &mdash; Output in the document %s. %s.', 'genesis' ), esc_html( '<meta name="robots" />' ), '<a href="http://yoast.com/articles/robots-meta-tags/" target="_blank">' . __( 'Read more about noarchive', 'genesis' ) . '</a>' ) . '</p>';

	get_current_screen()->add_help_tab( array(
		'id'       => 'genesis-inpost-seo',
		'title'    => __( 'Theme SEO Settings', 'genesis' ),
		'content'  => $content,
		'priority' => 12,
	) );

}

add_action( 'save_post', 'genesis_inpost_seo_save', 1, 2 );
/**
 * Save the SEO settings when we save a post or page.
 *
 * Some values get sanitized, the rest are pulled from identically named subkeys in the $_POST['genesis_seo'] array.
 *
 * @since 0.1.3
 *
 * @param int      $post_id Post ID.
 * @param stdClass $post    Post object.
 * @return null Return early if `genesis_seo` is not a key in `POST` data.
 */
function genesis_inpost_seo_save( $post_id, $post ) {

	if ( ! isset( $_POST['genesis_seo'] ) )
		return;

	// Merge user submitted options with fallback defaults.
	$data = wp_parse_args( $_POST['genesis_seo'], array(
		'_genesis_title'         => '',
		'_genesis_description'   => '',
		'_genesis_keywords'      => '',
		'_genesis_canonical_uri' => '',
		'redirect'               => '',
		'_genesis_noindex'       => 0,
		'_genesis_nofollow'      => 0,
		'_genesis_noarchive'     => 0,
	) );

	// Sanitize the title, description, and tags.
	foreach ( (array) $data as $key => $value ) {
		if ( in_array( $key, array( '_genesis_title', '_genesis_description', '_genesis_keywords' ) ) )
			$data[ $key ] = strip_tags( $value );
	}

	genesis_save_custom_fields( $data, 'genesis_inpost_seo_save', 'genesis_inpost_seo_nonce', $post );

}

add_action( 'admin_menu', 'genesis_add_inpost_scripts_box' );
/**
 * Register a new meta box to the post or page edit screen, so that the user can apply scripts on a per-post or
 * per-page basis.
 *
 * The scripts field was previously part of the SEO meta box, and was therefore hidden when an SEO plugin was active.
 *
 * @since 2.0.0
 *
 * @see genesis_inpost_scripts_box() Generates the content in the meta box.
 */
function genesis_add_inpost_scripts_box() {

	// If user doesn't have unfiltered html capability, don't show this box.
	if ( ! current_user_can( 'unfiltered_html' ) )
		return;

	foreach ( (array) get_post_types( array( 'public' => true ) ) as $type ) {
		if ( post_type_supports( $type, 'genesis-scripts' ) )
			add_meta_box( 'genesis_inpost_scripts_box', __( 'Scripts', 'genesis' ), 'genesis_inpost_scripts_box', $type, 'normal', 'low' );
	}

}

/**
 * Callback for in-post Scripts meta box.
 *
 * @since 2.0.0
 */
function genesis_inpost_scripts_box() {

	wp_nonce_field( 'genesis_inpost_scripts_save', 'genesis_inpost_scripts_nonce' );
	?>

	<table class="form-table">
	<tbody>

		<tr valign="top">
			<th scope="row"><label for="genesis_scripts"><strong><?php _e( 'Page-specific Scripts', 'genesis' ); ?></strong></label></th>
			<td>
				<p><textarea class="widefat" rows="4" name="genesis_seo[_genesis_scripts]" id="genesis_scripts"><?php echo esc_textarea( genesis_get_custom_field( '_genesis_scripts' ) ); ?></textarea></p>
				<p><?php printf( __( 'Suitable for custom tracking, conversion or other page-specific script. Must include %s tags.', 'genesis' ), genesis_code( 'script' ) ); ?></p>
			</td>
		</tr>

	</tbody>
	</table>

	<?php

}

add_action( 'save_post', 'genesis_inpost_scripts_save', 1, 2 );
/**
 * Save the Scripts settings when we save a post or page.
 *
 * @since 2.0.0
 *
 * @param int      $post_id Post ID.
 * @param stdClass $post    Post object.
 * @return null Return early if `genesis_seo` is not a key in `POST` data, or current user doesn't
 *              have `unfiltered_html` capability.
 */
function genesis_inpost_scripts_save( $post_id, $post ) {

	if ( ! isset( $_POST['genesis_seo'] ) )
		return;

	 // If user doesn't have unfiltered html capability, don't try to save.
	if ( ! current_user_can( 'unfiltered_html' ) )
		return;

	// Merge user submitted options with fallback defaults.
	$data = wp_parse_args( $_POST['genesis_seo'], array(
		'_genesis_scripts' => '',
	) );

	genesis_save_custom_fields( $data, 'genesis_inpost_scripts_save', 'genesis_inpost_scripts_nonce', $post );

}

add_action( 'admin_menu', 'genesis_add_inpost_layout_box' );
/**
 * Register a new meta box to the post or page edit screen, so that the user can set layout options on a per-post or
 * per-page basis.
 *
 * @since 0.2.2
 *
 * @see genesis_inpost_layout_box() Generates the content in the boxes
 *
 * @return null Return early if Genesis layouts are not supported.
 */
function genesis_add_inpost_layout_box() {

	if ( ! current_theme_supports( 'genesis-inpost-layouts' ) ) {
		return;
	}

	foreach ( (array) get_post_types( array( 'public' => true ) ) as $type ) {
		if ( post_type_supports( $type, 'genesis-layouts' ) )
			add_meta_box( 'genesis_inpost_layout_box', __( 'Layout Settings', 'genesis' ), 'genesis_inpost_layout_box', $type, 'normal', 'high' );
	}

}

/**
 * Callback for in-post layout meta box.
 *
 * @since 0.2.2
 */
function genesis_inpost_layout_box() {

	wp_nonce_field( 'genesis_inpost_layout_save', 'genesis_inpost_layout_nonce' );

	$layout = genesis_get_custom_field( '_genesis_layout' );
	?>

	<table class="form-table">
	<tbody>

	<?php if ( genesis_has_multiple_layouts() ) : ?>
		<tr valign="top">
			<th scope="row"><?php _e( 'Select Layout', 'genesis' ); ?></th>
			<td>
				<fieldset class="genesis-layout-selector">
					<legend class="screen-reader-text"><?php _e( 'Layout Settings', 'genesis' ); ?></legend>

					<p><input type="radio" name="genesis_layout[_genesis_layout]" class="default-layout" id="default-layout" value="" <?php checked( $layout, '' ); ?> /> <label class="default" for="default-layout"><?php printf( __( 'Default Layout set in <a href="%s">Theme Settings</a>', 'genesis' ), menu_page_url( 'genesis', 0 ) ); ?></label></p>
					<?php genesis_layout_selector( array( 'name' => 'genesis_layout[_genesis_layout]', 'selected' => $layout, 'type' => 'site' ) ); ?>

				</fieldset>
			</td>
		</tr>
	<?php endif; ?>

		<tr valign="top">
			<th scope="row"><label for="genesis_custom_body_class"><?php _e( 'Custom Body Class', 'genesis' ); ?></label></th>
			<td><p><input class="large-text" type="text" name="genesis_layout[_genesis_custom_body_class]" id="genesis_custom_body_class" value="<?php echo esc_attr( genesis_get_custom_field( '_genesis_custom_body_class' ) ); ?>" /></p></td>
		</tr>

		<tr valign="top">
			<th scope="row"><label for="genesis_custom_post_class"><?php _e( 'Custom Post Class', 'genesis' ); ?></label></th>
			<td><p><input class="large-text" type="text" name="genesis_layout[_genesis_custom_post_class]" id="genesis_custom_post_class" value="<?php echo esc_attr( genesis_get_custom_field( '_genesis_custom_post_class' ) ); ?>" /></p></td>
		</tr>

	</tbody>
	</table>

	<?php

}

add_action( 'save_post', 'genesis_inpost_layout_save', 1, 2 );
/**
 * Save the layout options when we save a post or page.
 *
 * Since there's no sanitizing of data, the values are pulled from identically named keys in $_POST.
 *
 * @since 0.2.2
 *
 * @param int      $post_id Post ID.
 * @param stdClass $post    Post object.
 * @return null Return early if `genesis_layout` is not a key in `POST` data.
 */
function genesis_inpost_layout_save( $post_id, $post ) {

	if ( ! isset( $_POST['genesis_layout'] ) )
		return;

	$data = wp_parse_args( $_POST['genesis_layout'], array(
		'_genesis_layout'            => '',
		'_genesis_custom_body_class' => '',
		'_genesis_post_class'        => '',
	) );

	$data = array_map( 'genesis_sanitize_html_classes', $data );

	genesis_save_custom_fields( $data, 'genesis_inpost_layout_save', 'genesis_inpost_layout_nonce', $post );

}
